//SCREEN-BOILERPLATE: Ladle Wall Overview
//Renders the ASPX grid table by fetching LAD.GetLadleWallOverview and filling cells.

define([
  'app',
  'js/templates-loader',
  'backgrid',
  'chart',
  'moment',
  'chartjs-date-fns-adapter',
  'modules/navigation-buttons/navigation-buttons',
  'modules/select2-module/select2-module',
  'modules/modal'
], function (app, T, Backgrid, ChartJS, moment, ChartJsDateFnsAdapter, NavigationButton, select2Module, Modal) {

  var Screen = { Models: {}, Views: {}, Collections: {} };

  // -------------------------
  // Collection
  // -------------------------
  Screen.Collections.ItemsColl = Backbone.Collection.extend({
    setDataColl: function (data) {
      this.set(data).trigger('fetch', this, data);
    }
  });

  // -------------------------
  // Model
  // -------------------------
  Screen.Models.Main = Backbone.Epoxy.Model.extend({
    defaults: {
      hasData: false,
      isLoading: false,
      selectedLadleId: null
    },

    initialize: function () {
      this.ladleWallOverview = new Screen.Collections.ItemsColl(); // one row per LadleId
    },

    fetchLadleWallOverview: function () {
      var that = this;
      var qp = new Core.Database.QueryParameters();

      this.set({ isLoading: true, hasData: false });

      Core.Json.CallProcedure(
        app.DatabaseNames.MES + '.LAD.GetLadleWallOverview',
        qp,
        {
          onSuccess: function (resp) {
            try {
              if (resp && resp.Table) {
                that.ladleWallOverview.setDataColl(resp.Table);
                that.trigger('ladle-wall-overview-fetched');
                that.set({ isLoading: false, hasData: true });
              } else {
                app.views.topMessages.showMessage('No ladle overview returned', { stay: 4000 });
                that.set({ isLoading: false, hasData: false });
                that.trigger('ladle-wall-overview-failed');
              }
            } catch (e) {
              console.error(e.stack || e);
              that.set({ isLoading: false, hasData: false });
              that.trigger('ladle-wall-overview-failed');
            }
          },
          onFailure: function (resp) {
            console.error(resp);
            that.set({ isLoading: false, hasData: false });
            app.views.topMessages.showMessage('Failed to load overview', { stay: 4000 });
            that.trigger('ladle-wall-overview-failed');
          },
          Secured: true
        },
        app.ConnectionStrings.app
      );

      return this;
    }
  });

    // safe keyBy for Underscore/Lodash variants
    function keyBy(arr, prop) {
    if (typeof _ !== 'undefined') {
        if (_.keyBy) return _.keyBy(arr, prop);                 
        if (_.indexBy) return _.indexBy(arr, prop);            
    }
    // Fallback
    return (arr || []).reduce(function (acc, item) {
        if (item && Object.prototype.hasOwnProperty.call(item, prop)) {
        acc[item[prop]] = item;
        }
        return acc;
    }, {});
    }


  // -------------------------
  // View
  // -------------------------
  Screen.Views.Main = Backbone.Epoxy.View.extend({
    template: 'ladle-wall-overview',
    id: 'ladle-wall-overview',
    title: 'Ladle Wall Overview',
    isCacheable: false,

    events: function () {
      return {
        'click #refreshBtn': this.onRefresh,
        'click #ladleWallGrid td[data-col^="Ladle-"]': this.onGridCellClick
      };
    },

    bindings: 'data-bind',
    subviews: null,

    initialize: function () {
      this.options.state = app.view_states.loading;
      this.options.onappend = (_.isFunction(this.options.onappend)) ? this.options.onappend : function () { };

      this.model = new Screen.Models.Main();
      this.bindEvents();
    },

    bindEvents: function () {
      this.listenTo(this.model, 'ladle-wall-overview-fetched', this.renderWallGrid);
      this.listenTo(this.model, 'ladle-wall-overview-failed',   this.endRefreshUI);
    },

    // -------------------------
    // Render lifecycle
    // -------------------------
    render: function (container) {
      var that = this;
      var thatContainer = (this.options.container) ? this.options.container : container;
      this.options.container = thatContainer;

      var customPath = '/app/pages/eaf/' + this.template + '/';

      T.render.call(
        this,
        this.template,
        function (tmp) {
          if (!that.options.i18n) that.options.i18n = {};

          app.getI18NJed(
            that,
            that.template,
            function (i18nJED) {
              that.options.i18n[that.template] = i18nJED;
              that.$el.html(tmp());

              that.applyBindings();
              that.append(thatContainer, that.$el);

              // Initial load
              that.showSkeleton();
              that.model.fetchLadleWallOverview();
            },
            true,
            customPath
          );
        },
        customPath
      );
    },

    append: function (container, el) {
      el = el || this.$el;

      if (this.options.state == app.view_states.loading || this.options.state == app.view_states.shown) {
        this.options.state = app.view_states.shown;
        container.append(el);
        this.options.onappend(this);
      } else if (this.options.state == app.view_states.hidden) {
        container.append(el);
      } else if (this.options.state == app.view_states.closed) {
        return;
      }
    },

    // -------------------------
    // Grid rendering
    // -------------------------
    renderWallGrid: function () {
        this.hideSkeleton();
      // Detect how many ladles to render  default = 15 (current in August 2025)
      var LADLES = 15;
        var data = this.model.ladleWallOverview.toJSON();
        if (data && data.length) {
        var maxId = Math.max.apply(null, data.map(function (r){ return +r.LadleId || 0; }));
        if (isFinite(maxId) && maxId > 0) LADLES = Math.max(LADLES, maxId);
        }

      // Map the UI row labels -> field names returned by the SP.
      // NOTE: "Campaign" is mapped to "LadleShell" here. Change if you want a different field.
      var ROW_FIELD = {
        "Heat Number":       "HeatId",
        "Tap Time":          "TapTime",
        "Campaign":          "Campaign",       
        "Plate Assembly":    "PlateAssembly",
        "Upper Nozzle":      "UpperNozzle",
        "Working Lining":    "WorkingLining",
        "Porous Plug North": "PorousPlugNorth",
        "Porous Plug South": "PorousPlugSouth"
      };

      // Index the returned rows by LadleId for quick lookup
      var byLadle = keyBy(this.model.ladleWallOverview.toJSON(), "LadleId");

      // Clear grid cells
      this.$("#ladleWallGrid [data-col]").empty();

      // Fill per-row/per-ladle cells
      var that = this;
      _.each(ROW_FIELD, function (field, rowLabel) {
        for (var lid = 1; lid <= LADLES; lid++) {
          var rowData = byLadle[lid];
          var val = rowData ? rowData[field] : "";

          // Simple formatters
          if (field === "TapTime" && val) {
            val = moment(val).format("YYYY-MM-DD HH:mm");
          }
          if (field === "StartTap" && val) {
            val = moment(val).format("YYYY-MM-DD HH:mm");
          }
          // Optional: format numbers with no trailing zeros, etc.

          that.$("#ladleWallGrid [data-row='" + rowLabel + "'][data-col='Ladle-" + lid + "']").text(val || "");
        }
      });

      // Optional: highlight currently selected ladle column
      this.highlightSelectedLadle();
      this.endRefreshUI();
    },

    highlightSelectedLadle: function () {
      var lid = this.model.get('selectedLadleId');
      var $cells = this.$("#ladleWallGrid td[data-col^='Ladle-'], #ladleWallGrid th");

      // clear previous
      $cells.removeClass('ring-1 ring-blue-400');

      if (lid != null) {
        this.$("#ladleWallGrid [data-col='Ladle-" + lid + "']").addClass('ring-1 ring-blue-400');
      }
    },

    // Helpers (inside Screen.Views.Main)
        showSkeleton: function () {
        this.$("#gridBody").addClass("hidden");
        this.$("#gridSkeleton").removeClass("hidden");
        },
        hideSkeleton: function () {
        this.$("#gridSkeleton").addClass("hidden");
        this.$("#gridBody").removeClass("hidden");
        },

        startRefreshUI: function () {
  // remember selection so we restore highlight after reload
  this._prevSelected = this.model.get('selectedLadleId');

  // show skeleton & disable button
  this.showSkeleton();
  var $btn = this.$('#refreshBtn');
  $btn.prop('disabled', true)
      .addClass('opacity-60 cursor-not-allowed')
      .data('label', $btn.text())
      .html('<span class="inline-flex items-center gap-2"><i class="fa fa-refresh fa-spin"></i> Refreshing…</span>');
},

endRefreshUI: function () {
  var $btn = this.$('#refreshBtn');
  $btn.prop('disabled', false)
      .removeClass('opacity-60 cursor-not-allowed')
      .text($btn.data('label') || 'Refresh');

  // restore highlight on the previously selected ladle
  if (this._prevSelected != null) {
    this.model.set('selectedLadleId', this._prevSelected);
    this.highlightSelectedLadle();
  }
},



    // -------------------------
    // Events
    // -------------------------
    onRefresh: function () {
      if (this.$('#refreshBtn').prop('disabled')) return;
      this.startRefreshUI();
      this.model.fetchLadleWallOverview();
    },

    onSave: function () {
      // No save on this overview grid (read-only) — keep for future actions
      app.views.topMessages.showMessage('Nothing to save on overview.', { stay: 2500 });
    },

    onGridCellClick: function (e) {
      var col = e.currentTarget.getAttribute('data-col'); // "Ladle-7"
      var ladleId = parseInt(col.split('-')[1], 10);
      if (isFinite(ladleId)) {
        this.model.set('selectedLadleId', ladleId);
        this.highlightSelectedLadle();

        // Optional: if you want to load the detailed pane you already have:
        // this.model.fetchLadleInfo(ladleId);
      }
    },

    // -------------------------
    // View lifecycle helpers
    // -------------------------
    close: function () {
      this.options.state = app.view_states.closed;
      this.remove();
      this.unbind();
      this.stopListening();
    },
    show: function () {
      this.options.state = app.view_states.shown;
      this.$el.show();
      this.bindEvents();
    },
    hide: function () {
      this.options.state = app.view_states.hidden;
      this.$el.hide();
      this.unbind();
      this.stopListening();
    }
  });

  // Required, return the module for AMD compliance.
  return Screen;
});
